/*:
 * @target MZ
 * @plugindesc Live2Dモデルの特定部位に円形モザイクをかける（追従・スイッチ制御対応） [ver: PluginCommand対応]
 * @author ChatGPT
 *
 * @command AddMosaic
 * @text モザイク追加
 * @desc モデルに円形モザイクを追加し、指定Drawableに追従させる
 *
 * @arg modelName
 * @text モデルID
 * @type string
 * @desc Live2DモデルのID（例：chisa_01）
 *
 * @arg drawableId
 * @text Drawable ID
 * @type string
 * @desc モザイクをかけるアートメッシュID（例：Part_Mosaic）
 *
 * @arg radius
 * @text モザイクの半径(px)
 * @type number
 * @default 30
 *
 * @arg pixelSize
 * @text モザイクのピクセルサイズ
 * @type number
 * @default 11
 *
 * @arg switchId
 * @text 有効スイッチID
 * @type switch
 * @default 0
 *
 * @help Live2DMosaicFilter.js
 *
 * Live2DInterfaceMZ.js にて表示中のLive2Dモデルに対し、
 * 特定のアートメッシュ（例："Part_Mosaic"）に追従する円形モザイクを表示します。
 * モザイクの有効／無効はツクール内のスイッチで制御可能です。
 *
 * プラグインコマンド：AddMosaic
 * - モデルID、Drawable ID、モザイクの半径・サイズ・スイッチIDを指定します。
 *
 * スクリプトでも以下のように呼び出せます：
 * Live2DMosaicFilter.add({
 *     modelName: "chisa_01",
 *     drawableId: "Part_Mosaic",
 *     radius: 30,
 *     pixelSize: 11,
 *     switchId: 10
 * });
 */

var Live2DMosaicFilter = (() => {
    const mosaicEntries = [];

    const plugin = {
        add({modelName, drawableId, radius = 30, pixelSize = 11, switchId = 0}) {
            mosaicEntries.push({ modelName, drawableId, radius, pixelSize, switchId, sprite: null, mask: null });
        },

        update() {
            for (const entry of mosaicEntries) {
                const modelSprite = SceneManager._scene.children.find(child =>
                    child._live2dModel && child._live2dModel.modelID === entry.modelName);
                if (!modelSprite || !$gameSwitches.value(entry.switchId)) {
                    if (entry.sprite) entry.sprite.visible = false;
                    continue;
                }

                const internal = modelSprite._live2dModel.internalModel;
                const drawables = internal.drawables;
                const idx = drawables.ids.indexOf(entry.drawableId);
                if (idx < 0) continue;

                const verts = drawables.vertexPositions[idx];
                let x = 0, y = 0, len = verts.length / 2;
                for (let i = 0; i < verts.length; i += 2) {
                    x += verts[i];
                    y += verts[i + 1];
                }
                x /= len;
                y /= len;

                const pos = modelSprite._live2dModel.toGlobal(new PIXI.Point(x, y));

                if (!entry.sprite) {
                    const r = entry.radius;
                    const g = new PIXI.Graphics();
                    g.beginFill(0xffffff);
                    g.drawCircle(0, 0, r);
                    g.endFill();

                    const rt = PIXI.RenderTexture.create({width: r*2, height: r*2});
                    const s = new PIXI.Sprite(rt);
                    s.filters = [new PIXI.filters.PixelateFilter(entry.pixelSize)];
                    s.mask = g;
                    s.addChild(g);
                    SceneManager._scene.addChild(s);
                    entry.sprite = s;
                    entry.mask = g;
                }

                entry.sprite.visible = true;
                entry.sprite.x = pos.x - entry.radius;
                entry.sprite.y = pos.y - entry.radius;
                entry.mask.x = entry.radius;
                entry.mask.y = entry.radius;

                const r = entry.radius;
                const rt = entry.sprite.texture;
              // Live2Dモデルの実体を取得
const l2d = modelSprite._live2dModel;

if (l2d) {
  const r = entry.radius;
  const bounds = new PIXI.Rectangle(pos.x - r, pos.y - r, r * 2, r * 2);

  // Live2DモデルをRenderTextureに描画（この範囲にモザイク処理がかかる）
  SceneManager._scene.app.renderer.render(l2d, rt, true, bounds);
}

            }
        }
    };

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        plugin.update();
    };

    PluginManager.registerCommand("Live2DMosaicFilter", "AddMosaic", args => {
        plugin.add({
            modelName: args.modelName,
            drawableId: args.drawableId,
            radius: Number(args.radius || 30),
            pixelSize: Number(args.pixelSize || 11),
            switchId: Number(args.switchId || 0)
        });
    });

    return plugin;
})();
